import 'dart:html';

import 'package:rad/rad.dart';
import 'package:rad/widgets_html.dart';

import 'package:photogram_admin_cp/import/data.dart';
import 'package:photogram_admin_cp/import/core.dart';

class SettingItem {
  int intId;
  String settingKey;
  FieldType type;
  String leading;
  String? description;
  String? placeholder;
  Map<String, String>? allowedValues;

  SettingItem({
    required this.intId,
    required this.settingKey,
    required this.type,
    required this.leading,
    this.description,
    this.placeholder,
    this.allowedValues,
  });
}

class FormEditorPage extends StatefulWidget {
  final String? heading;
  final String? description;

  final List<SettingItem> items;

  const FormEditorPage({
    Key? key,
    required this.items,
    this.heading,
    this.description,
  }) : super(key: key);

  @override
  _FormEditorPageState createState() => _FormEditorPageState();
}

class _FormEditorPageState extends State<FormEditorPage> with AppActiveContentMixin {
  final _settingItems = <SettingItem>[];
  final _settingModels = <String, SettingModel>{};

  var _isUnsaved = false;
  var _isSaveInProgress = false;
  var _saveError = '';
  var _saveErrorAtKey = '';

  @override
  void onLoadEvent() {
    _settingItems.addAll(widget.items);

    _loadModels();
  }

  @override
  build(context) {
    return Division(
      className: 'content-forced animate__animated animate__fadeIn',
      children: [
        // page heading and description

        if (null != widget.heading) Heading1(className: 'content-title font-weight-medium', innerText: widget.heading),

        if (null != widget.description) Division(className: 'text-muted font-size-12', innerText: widget.description),

        // fields

        for (var item in _settingItems) _getEditingField(item),

        // break lines to prevent overlay

        const LineBreak(),
        const LineBreak(),
        const LineBreak(),
        const LineBreak(),
        const LineBreak(),

        // unsaved detector

        if (_isUnsaved)
          Division(
            style: 'position: fixed; bottom: 80px; max-width: 665px; width:100%;',
            children: [
              Division(
                className: 'alert animate__animated animate__slideInUp'
                    ' ${_saveError.isNotEmpty ? 'alert-danger' : 'alert-secondary'} ',
                children: [
                  Division(
                    className: 'clearfix',
                    children: [
                      Division(
                        className: 'float-left d-inline-block font-size-14 font-weight-medium',
                        innerText: _saveError.isNotEmpty ? 'Something went wrong' : 'Unsaved Changes',
                      ),
                      Division(
                        className: 'float-right d-inline-block ',
                        child: Button(
                          className: 'btn alt-dm',
                          onClick: _saveSettings,
                          disabled: _isSaveInProgress,
                          innerText: _isSaveInProgress ? 'Saving...' : (_saveError.isNotEmpty ? 'Retry' : 'Save'),
                        ),
                      ),
                    ],
                  ),
                  Division(
                    className: 'font-size-12 text-muted my-5',
                    innerText: _saveError.isNotEmpty ? _saveError : "Careful - You've unsaved changes",
                  ),
                ],
              ),
            ],
          ),
      ],
    );
  }

  Widget _getEditingField(SettingItem item) {
    var error = (_saveErrorAtKey.isNotEmpty && item.settingKey == _saveErrorAtKey) ? 'is-invalid' : '';

    switch (item.type) {
      case FieldType.switchable:
        void switchableOnChange(EmittedEvent event) {
          var checkbox = event.target as CheckboxInputElement;
          var isChecked = checkbox.checked ?? false;
          _settingModels[item.settingKey]!.update(isChecked ? '1' : '0');
          setState(() => _refreshUnsaveState());
        }

        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              className: 'clearfix',
              children: [
                Division(
                  className: 'float-left d-inline-block font-size-14 font-weight-medium',
                  innerText: item.leading,
                ),
                Division(
                  className: 'float-right d-inline-block',
                  child: Division(
                    className: 'custom-switch',
                    children: [
                      InputCheckBox(
                        id: item.settingKey,
                        checked: _settingModels[item.settingKey]!.isOn,
                        onChange: switchableOnChange,
                      ),
                      Label(forAttribute: item.settingKey),
                    ],
                  ),
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: item.description,
            ),
          ],
        );

      case FieldType.text:
        void textOnChange(EmittedEvent event) {
          var inputText = event.target as InputElement;
          _settingModels[item.settingKey]!.update(inputText.value ?? '');
          setState(() => _refreshUnsaveState());
        }

        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              children: [
                Division(
                  className: 'font-size-14 font-weight-medium',
                  innerText: item.leading,
                ),
                InputText(
                  id: item.settingKey,
                  className: 'form-control my-10 px-10 py-15 $error',
                  placeholder: item.placeholder,
                  value: _settingModels[item.settingKey]!.updatedValue,
                  onInput: textOnChange,
                  onChange: textOnChange,
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: item.description,
            ),
          ],
        );

      case FieldType.select:
        void selectOnChange(EmittedEvent event) {
          var inputText = event.target as SelectElement;
          _settingModels[item.settingKey]!.update(inputText.value ?? '');
          setState(() => _refreshUnsaveState());
        }

        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              children: [
                Division(
                  className: 'font-size-14 font-weight-medium',
                  innerText: item.leading,
                ),
                Select(
                  id: item.settingKey,
                  className: 'form-control my-10 $error',
                  onChange: selectOnChange,
                  children:
                      _buildSelectOptions(item.allowedValues ?? {}, _settingModels[item.settingKey]!.updatedValue),
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: item.description,
            ),
          ],
        );

      case FieldType.textarea:
        void textareaOnChange(EmittedEvent event) {
          var inputText = event.target as TextAreaElement;
          _settingModels[item.settingKey]!.update(inputText.value ?? '');
          setState(() => _refreshUnsaveState());
        }

        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              children: [
                Division(
                  className: 'font-size-14 font-weight-medium',
                  innerText: item.leading,
                ),
                Division(
                  className: 'custom-switch',
                  children: [
                    TextArea(
                      id: item.settingKey,
                      className: 'form-control my-10 px-10 py-15 $error',
                      placeholder: item.placeholder,
                      innerText: _settingModels[item.settingKey]!.updatedValue,
                      onInput: textareaOnChange,
                      onChange: textareaOnChange,
                    )
                  ],
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: item.description,
            ),
          ],
        );
    }
  }

  List<Widget> _buildSelectOptions(Map<String, String> valuesMap, String selectedValue) {
    var options = <Widget>[];

    print(selectedValue);

    valuesMap.forEach((key, value) {
      print('building:  $key : $value');

      options.add(
        Option(
          value: value,
          innerText: key,
          selected: value == selectedValue,
        ),
      );
    });

    return options;
  }

  void _loadModels() {
    for (var item in widget.items) {
      var settingModel = activeContent.read<SettingModel>(item.intId);

      if (null == settingModel) {
        throw "Something went wrong. Setting model for #${item.intId} of (${item.settingKey}) not found.";
      }

      _settingModels[item.settingKey] = settingModel;
    }
  }

  void _refreshUnsaveState() {
    var isAnyModelChanged = false;

    for (var model in _settingModels.values) {
      if (model.isChanged) {
        isAnyModelChanged = true;

        break;
      }
    }

    if (isAnyModelChanged && !_isUnsaved) {
      _isUnsaved = true;
      _saveError = '';
      _saveErrorAtKey = '';

      return;
    }

    if (!isAnyModelChanged && _isUnsaved) {
      _isUnsaved = false;
      _saveError = '';
      _saveErrorAtKey = '';

      return;
    }
  }

  void _saveSettings(EmittedEvent event) async {
    if (_isSaveInProgress) return;

    setState(() {
      _isSaveInProgress = true;
    });

    try {
      var apiRepo = activeContent.apiRepository;

      var settings = <String, dynamic>{};

      for (var item in _settingModels.values) {
        settings[item.id] = item;
      }

      var responseModel = await apiRepo.preparedRequest(
        requestType: REQ_TYPE_ADMIN_SETTINGS_SAVE,
        requestData: {
          SettingTable.tableName: {SettingTable.id: settings},
        },
      );

      if (responseModel.contains(AdminSettinSaveResultsDTO.dtoName)) {
        var resultDTO = AdminSettinSaveResultsDTO.fromJson(responseModel.first(AdminSettinSaveResultsDTO.dtoName));

        if (resultDTO.isDTO) {
          _saveError = resultDTO.message;
          _saveErrorAtKey = resultDTO.settingKey;

          return setState(() {
            _isSaveInProgress = false;
          });
        }
      }

      activeContent.handleResponse(responseModel);
    } catch (e) {
      AppLogger.exception(e);
    }

    setState(() {
      _refreshUnsaveState();

      _isSaveInProgress = false;
    });
  }
}
